/*
 * Decompiled with CFR 0.152.
 */
package org.hsqldb.persist;

import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import org.hsqldb.Database;
import org.hsqldb.error.Error;
import org.hsqldb.lib.ArrayUtil;
import org.hsqldb.lib.java.JavaSystem;
import org.hsqldb.persist.EventLogInterface;
import org.hsqldb.persist.RandomAccessInterface;

final class RAFileNIO
implements RandomAccessInterface {
    private final EventLogInterface logger;
    private final boolean readOnly;
    private final long maxLength;
    private long fileLength;
    private RandomAccessFile file;
    private FileDescriptor fileDescriptor;
    private MappedByteBuffer buffer;
    private long bufferPosition;
    private int bufferLength;
    private long currentPosition;
    private FileChannel channel;
    private boolean buffersModified;
    private MappedByteBuffer[] buffers = new MappedByteBuffer[0];
    private static final String JVM_ERROR = "NIO access failed";
    static final int largeBufferScale = 24;
    static final int largeBufferSize = 0x1000000;
    static final long largeBufferMask = -16777216L;

    RAFileNIO(EventLogInterface logger, String name, boolean readOnly, long requiredLength, long maxLength) throws IOException {
        this.logger = logger;
        this.maxLength = maxLength;
        File tempFile = new File(name);
        if (readOnly) {
            requiredLength = tempFile.length();
        } else {
            if (tempFile.length() > requiredLength) {
                requiredLength = tempFile.length();
            }
            requiredLength = ArrayUtil.getBinaryNormalisedCeiling(requiredLength, 24);
        }
        this.file = new RandomAccessFile(name, readOnly ? "r" : "rw");
        this.readOnly = readOnly;
        this.channel = this.file.getChannel();
        this.fileDescriptor = this.file.getFD();
        if (!this.ensureLength(requiredLength)) {
            this.close();
            IOException io = new IOException("NIO buffer allocation failed");
            throw io;
        }
        this.buffer = this.buffers[0];
        this.bufferLength = this.buffer.limit();
        this.bufferPosition = 0L;
        this.currentPosition = 0L;
    }

    @Override
    public long length() throws IOException {
        try {
            return this.file.length();
        }
        catch (IOException e) {
            this.logger.logWarningEvent(JVM_ERROR, e);
            throw e;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void seek(long newPos) throws IOException {
        try {
            this.positionBufferSeek(newPos);
            this.buffer.position((int)(newPos - this.bufferPosition));
        }
        catch (IllegalArgumentException e) {
            this.logger.logWarningEvent(JVM_ERROR, e);
            IOException io = JavaSystem.toIOException(e);
            throw io;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public long getFilePointer() throws IOException {
        try {
            return this.currentPosition;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public int read() throws IOException {
        try {
            byte value = this.buffer.get();
            this.positionBufferMove(1);
            return value;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void read(byte[] b, int offset, int length) throws IOException {
        try {
            do {
                this.checkBuffer();
                long transferLength = this.bufferPosition + (long)this.bufferLength - this.currentPosition;
                if (transferLength > (long)length) {
                    transferLength = length;
                }
                this.buffer.get(b, offset, (int)transferLength);
                this.positionBufferMove((int)transferLength);
                length = (int)((long)length - transferLength);
                offset = (int)((long)offset + transferLength);
            } while (length != 0);
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public int readInt() throws IOException {
        try {
            int value = this.buffer.getInt();
            this.positionBufferMove(4);
            return value;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public long readLong() throws IOException {
        try {
            long value = this.buffer.getLong();
            this.positionBufferMove(8);
            return value;
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void write(byte[] b, int offset, int length) throws IOException {
        try {
            this.buffersModified = true;
            do {
                this.checkBuffer();
                long transferLength = this.bufferPosition + (long)this.bufferLength - this.currentPosition;
                if (transferLength > (long)length) {
                    transferLength = length;
                }
                this.buffer.put(b, offset, (int)transferLength);
                this.positionBufferMove((int)transferLength);
                length = (int)((long)length - transferLength);
                offset = (int)((long)offset + transferLength);
            } while (length != 0);
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void writeInt(int i) throws IOException {
        try {
            this.buffersModified = true;
            this.buffer.putInt(i);
            this.positionBufferMove(4);
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void writeLong(long i) throws IOException {
        try {
            this.buffersModified = true;
            this.buffer.putLong(i);
            this.positionBufferMove(8);
        }
        catch (Throwable t) {
            this.logger.logWarningEvent(JVM_ERROR, t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.logger.logDetailEvent("NIO file close, size: " + this.fileLength);
            this.buffer = null;
            this.channel = null;
            for (int i = 0; i < this.buffers.length; ++i) {
                this.unmap(this.buffers[i]);
                this.buffers[i] = null;
            }
            this.file.close();
        }
        catch (Throwable t) {
            this.logger.logWarningEvent("NIO buffer close error", t);
            IOException io = JavaSystem.toIOException(t);
            throw io;
        }
    }

    @Override
    public boolean isReadOnly() {
        return this.readOnly;
    }

    @Override
    public boolean ensureLength(long newLength) {
        if (newLength > this.maxLength) {
            return false;
        }
        while (newLength > this.fileLength) {
            if (this.enlargeFile(newLength)) continue;
            return false;
        }
        return true;
    }

    private boolean enlargeFile(long newFileLength) {
        try {
            FileChannel.MapMode mapMode;
            long newBufferLength = newFileLength;
            if (!this.readOnly) {
                newBufferLength = 0x1000000L;
            }
            FileChannel.MapMode mapMode2 = mapMode = this.readOnly ? FileChannel.MapMode.READ_ONLY : FileChannel.MapMode.READ_WRITE;
            if (!this.readOnly && this.file.length() < this.fileLength + newBufferLength) {
                this.file.seek(this.fileLength + newBufferLength - 1L);
                this.file.writeByte(0);
            }
            MappedByteBuffer[] newBuffers = new MappedByteBuffer[this.buffers.length + 1];
            MappedByteBuffer newBuffer = this.channel.map(mapMode, this.fileLength, newBufferLength);
            System.arraycopy(this.buffers, 0, newBuffers, 0, this.buffers.length);
            newBuffers[this.buffers.length] = newBuffer;
            this.buffers = newBuffers;
            this.fileLength += newBufferLength;
            this.logger.logDetailEvent("NIO buffer instance, file size " + this.fileLength);
        }
        catch (Throwable e) {
            this.logger.logDetailEvent("NOI buffer allocate failed, file size " + newFileLength);
            return false;
        }
        return true;
    }

    @Override
    public boolean setLength(long newLength) {
        if (newLength > this.fileLength) {
            return this.enlargeFile(newLength);
        }
        try {
            this.seek(0L);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return true;
    }

    public Database getDatabase() {
        return null;
    }

    @Override
    public void synch() {
        int i;
        boolean error = false;
        int errIndex = 0;
        for (i = 0; i < this.buffers.length; ++i) {
            try {
                this.buffers[i].force();
                continue;
            }
            catch (Throwable t) {
                this.logger.logWarningEvent("NIO buffer force error: pos " + i * 0x1000000 + " ", t);
                if (!error) {
                    errIndex = i;
                }
                error = true;
            }
        }
        if (error) {
            for (i = errIndex; i < this.buffers.length; ++i) {
                try {
                    this.buffers[i].force();
                    continue;
                }
                catch (Throwable t) {
                    this.logger.logWarningEvent("NIO buffer force error " + i * 0x1000000 + " ", t);
                }
            }
        }
        try {
            this.fileDescriptor.sync();
            this.buffersModified = false;
        }
        catch (Throwable t) {
            this.logger.logSevereEvent("NIO RA file sync error ", t);
            throw Error.error(t, 452, null);
        }
    }

    private void positionBufferSeek(long offset) {
        if (offset < this.bufferPosition || offset >= this.bufferPosition + (long)this.bufferLength) {
            this.setCurrentBuffer(offset);
        }
        this.buffer.position((int)(offset - this.bufferPosition));
        this.currentPosition = offset;
    }

    private void positionBufferMove(int relOffset) {
        long offset = this.currentPosition + (long)relOffset;
        if (offset >= this.bufferPosition + (long)this.bufferLength) {
            this.setCurrentBuffer(offset);
        }
        this.buffer.position((int)(offset - this.bufferPosition));
        this.currentPosition = offset;
    }

    private void setCurrentBuffer(long offset) {
        int bufferIndex = (int)(offset >> 24);
        if (bufferIndex == this.buffers.length) {
            bufferIndex = this.buffers.length - 1;
            this.bufferPosition = (long)bufferIndex * 0x1000000L;
            this.buffer = this.buffers[bufferIndex];
            return;
        }
        this.buffer = this.buffers[bufferIndex];
        this.bufferPosition = offset & 0xFFFFFFFFFF000000L;
    }

    private void checkBuffer() {
        int bufferIndex = (int)(this.currentPosition >> 24);
        if (this.currentPosition != this.bufferPosition + (long)this.buffer.position()) {
            this.buffer = this.buffers[bufferIndex];
            this.bufferPosition = this.currentPosition & 0xFFFFFFFFFF000000L;
            this.buffer.position((int)(this.currentPosition - this.bufferPosition));
        } else if (this.buffer != this.buffers[bufferIndex]) {
            this.buffer = this.buffers[bufferIndex];
        }
    }

    private void unmap(MappedByteBuffer buffer) throws IOException {
        if (buffer == null) {
            return;
        }
        try {
            Method cleanerMethod = buffer.getClass().getMethod("cleaner", new Class[0]);
            cleanerMethod.setAccessible(true);
            Object cleaner = cleanerMethod.invoke((Object)buffer, new Object[0]);
            Method clearMethod = cleaner.getClass().getMethod("clean", new Class[0]);
            clearMethod.invoke(cleaner, new Object[0]);
        }
        catch (InvocationTargetException e) {
        }
        catch (NoSuchMethodException e) {
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }
}

